/******************************************************************************
 * %Id: $
 *
 * FileName		:dev-internal.c
 *
 * Description	:Communication Driver (Common part)
 *
 *
 * Copyright	:Panasonic Corporation.
 *
 *****************************************************************************/
#ifndef DEV_INTERNAL_C
#define DEV_INTERNAL_C
#define __NO_VERSION__ /* Must be first in all but one driver source file! */

#include "dev-share.h"
#include "iosc-internal-dev.h"
#include <linux/module.h>
#include <asm/pgtable.h>
#include <asm/current.h>
#include <linux/sched.h>
#include <linux/iosc/iosc-dev.h>
#include <linux/iosc/iosc-devices.h>

#include <linux/fs.h>

#include <asm/mman.h>
#include <asm/exmem.h>
#include "iprq.h"
#include "iprq_common.h"
#include "iprq_ext.h"

#define SKIP_RTOS_CHECK

static int iosc_ioc_malloc(int arg);
static int iosc_ioc_free(int arg);
static int iosc_ioc_attach_mem(int arg);
static int iosc_ioc_detach_mem(int arg);
static int iosc_ioc_ping_proc( struct iosc_port_t *self );
static int iosc_ioc_sig_proc( struct iosc_port_t *self, int arg );
static int iosc_ioc_getid_proc( struct iosc_port_t *self );
static int iosc_ioc_releaseid_proc( struct iosc_port_t *self );
static int iosc_ioc_getcmd_proc( struct iosc_port_t *self, struct iosc_read_packet_t *buf );
static int iosc_ioc_getdata_proc( struct iosc_port_t *self, void *arg, size_t size );
static int iosc_cmd_return_proc( struct iosc_port_t *self, unsigned int cmd, void *arg );
static int iosc_cmd_error_proc( struct iosc_port_t *self, unsigned int cmd, void *arg );
static int iosc_cmd_data_read_proc( struct iosc_port_t *self, unsigned int cmd, void *arg );
static int iosc_cmd_data_write_proc( struct iosc_port_t *self, unsigned int cmd, void *arg );
static int iosc_cmd_send_proc( struct iosc_port_t *self, unsigned int cmd, void *arg );

static void iosc_message( struct iosc_port_t *self, char *method, char *fmt, ... );
static inline int noncached_address(unsigned long addr);

/* typedef of method functions */

typedef ssize_t (*read_method_t) (struct file *, char *, size_t, loff_t *);
typedef ssize_t (*write_method_t) (struct file *, const char *, size_t, loff_t *);
typedef int (*open_method_t) (struct inode *, struct file *);
typedef int (*close_method_t) (struct inode *, struct file *);
typedef int (*ioctl_method_t) (struct inode *, struct file *, unsigned int, unsigned long);
typedef unsigned int (*poll_method_t) (struct file *, struct poll_table_struct *);
typedef int (*mmap_method_t) (struct file *, struct vm_area_struct *);
typedef int (*fsync_method_t) (struct file *, struct dentry *, int datasync);
typedef loff_t (*llseek_method_t) (struct file *, loff_t, int);

/* local function */
static int devbrg_local_init(struct iosc_dev_t *device);
static void devbrg_local_exit(struct iosc_dev_t *device);

static int devbrg_local_open (struct inode *inode, struct file *filp);
static int devbrg_local_release (struct inode * inode, struct file * filp);
static ssize_t devbrg_local_read(struct file *, char *, size_t, loff_t *);
static ssize_t devbrg_local_write(struct file *, const char *, size_t, loff_t *);
static int devbrg_local_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static int devbrg_local_poll(struct file *, struct poll_table_struct *);

/* static variables (for module information) */
struct iosc_dev_t *iosc_dev[DEVICE_NR];
/* dev ǡ static ˻ͳ
 *   (kamllocưŪݤʤfilp->private_dataǻʤͳ)
 * ưŪǡˤȡopen˳ݡrelease˲Ȥʤ롣
 * iosc-dev RTOS-LinuxĤΥǥХopenȤ
 * ֤ѤΤǤϤʤäơstaticѿˤääѤʤ
 * Թ礬ɤ
 */
extern struct iosc_dev_t *iosc_dev[256/*FIXME*/];

static struct mutex getid_mutex[DEVICE_NR];
static struct mutex open_mutex[DEVICE_NR];
static struct mutex stat_mutex[DEVICE_NR];

#define IOSC_DEFINE_DEVICE_BRIDGE(NAME, L_MAJOR, R_MAJOR, L_TYPE, R_TYPE, N_MINOR, MMAP_BASE) \
static struct iosc_dev_t NAME ## _iosc_dev;		\
static struct iosc_port_t NAME ## _port[N_MINOR];	\
static int __init init_##NAME(void)			\
{							\
	/*memset(&NAME ## _iosc_dev, 0, sizeof(NAME ## _iosc_dev));*/	\
	NAME ## _iosc_dev.module_name = #NAME;		\
	NAME ## _iosc_dev.mmap_base = MMAP_BASE;	\
	NAME ## _iosc_dev.n_minor = N_MINOR;		\
	NAME ## _iosc_dev.major = L_MAJOR;		\
	NAME ## _iosc_dev.type = L_TYPE;		\
	NAME ## _iosc_dev.name = "l" #NAME;		\
	NAME ## _iosc_dev.port = NAME ## _port;		\
	iosc_dev[L_MAJOR] = &NAME ## _iosc_dev;		\
	mutex_init(&getid_mutex[L_MAJOR]);		\
	mutex_init(&open_mutex[L_MAJOR]);		\
	mutex_init(&stat_mutex[L_MAJOR]);		\
	return devbrg_local_init(iosc_dev[L_MAJOR]);	\
}							\
static void __exit exit_##NAME(void){			\
	devbrg_local_exit(iosc_dev[L_MAJOR]);		\
	return;						\
}

/* file operation structure */

static struct file_operations l_fops = {
  read:    (read_method_t) devbrg_local_read,
  write:   (write_method_t)devbrg_local_write,
  open:    (open_method_t) devbrg_local_open,
  release: (close_method_t)devbrg_local_release,
  ioctl:   (ioctl_method_t)devbrg_local_ioctl,
  poll:    (poll_method_t) devbrg_local_poll,
};

/*
 * static int devbrg_local_init(struct iosc_dev_t *device)
 *-----------------------------------------------------------------------------
 * function: initialize module
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
static int devbrg_local_init(struct iosc_dev_t *device)
{
    int result;
    int port;

	iosc_msg("%s version " IOSC_VERSION "(size=%d)\n", device->module_name, sizeof(*device) + sizeof(struct iosc_port_t) * device->n_minor);

	result = register_chrdev(device->major, device->name, &l_fops);
	if (result < 0) {
		iosc_err("iosc-dev: can't register %s (major=%d).\n", device->name, device->major);
		goto L_ERROR_REG_LINUX;
	}
//    for( os = 0; os < IOSC_MAX_OS; ++os ){
	for (port = 0; port < device->n_minor; ++port) {
		device->port[port].major = device->major;
		device->port[port].minor = port;
		device->port[port].mmap_base = device->mmap_base;
		device->port[port].type = device->type;
		device->port[port].lock_flag = 0;
		device->port[port].open_flag = 0;
		init_waitqueue_head(&device->port[port].open_wait);
		init_packet_queue(&device->port[port].request);
		init_packet_queue(&device->port[port].reply);
		init_packet_queue(&device->port[port].fifo);
		init_datablock(device->port[port].block, IOSC_MAX_BLOCK);
		device->port[port].stat.open_count = 0;
		device->port[port].stat.release_count = 0;
		device->port[port].stat.read_count = 0;
		device->port[port].stat.write_count = 0;
		device->port[port].stat.mmap_count = 0;
		device->port[port].stat.ioctl_count = 0;
		device->port[port].stat.poll_count = 0;
		device->port[port].stat.read_byte = 0;
		device->port[port].stat.write_byte = 0;
	}
    return 0;
L_ERROR_REG_LINUX:
    return result;

}

/*
 * static void devbrg_local_exit(struct iosc_dev_t *device)
 *-----------------------------------------------------------------------------
 * function: exit module
 * argument: nothing
 * return  : nothing
 * comment :
 */
static void devbrg_local_exit(struct iosc_dev_t *device)
{
	int port;
	printk("in exit-func of module %s\n", device->module_name);

//    destroy_packet_pool( device->ctl );
//    destroy_packet_pool( device->data );
//    destroy_packet_pool( device->fifo );
	for (port = 0; port < device->n_minor; ++port) {
		clean_datablock(device->port[port].block, IOSC_MAX_BLOCK);
	}

	unregister_chrdev(device->major, device->name);
//    unregister_chrdev(dev.os[RTOS].major, dev.os[RTOS].name);
}


/*****************************************************************
 * Method functions for Linux-side device
 *****************************************************************/

/*
 * [Method] open
 * RTOS¦ΥǥХե뤬ץ󤵤Ƥʤ硢ץ󤵤ޤWAIT
 * @inode   iΡ
 * @filp    ե빽¤
 * RETURN: 0
 *         -ENOENT:ǥХ¸ߤʤ
 *         -EINVALO_RDWRʳ°ǥץ󤵤줿
 *         -EBUSYǥХϤǤ˥ץ󤵤Ƥ
 */
static int devbrg_local_open(struct inode *inode, struct file *filp)
{
    int ret = 0;
    int major = MAJOR( inode->i_rdev );
    int minor = MINOR( inode->i_rdev );
    struct iosc_port_t *self;

	if (!iosc_dev[major]) {
		return -ENOENT;
	}
	if (minor >= iosc_dev[major]->n_minor) {
		return -ENOENT;
	}

	self = (struct iosc_port_t *)&iosc_dev[major]->port[minor];

    iosc_message( self, "open", NULL );

    if ((filp->f_flags & O_RDWR) == 0) { /* O_RDONLY/O_WRONLYǤϳʤ */
        return -EINVAL;
    }

    mutex_lock( &open_mutex[major] );
    if( self->open_flag != 0 )
    {
        mutex_unlock( &open_mutex[major] );
        return -EBUSY;
    }
    if ( ( self->lock_flag != 0 ) && ( self->lock_flag != current->pid ) )
    {
        mutex_unlock( &open_mutex[major] );
        return -EBUSY;
    }
	
    mutex_init(&self->mutex);
    self->open_flag = 1;
    mutex_unlock( &open_mutex[major] );

#if 0
    if ( self->os == LINUX )
    {
        /* Linuxξ硢RTOSΥץԤ */
        wait_event_interruptible( self->open_wait, self->buddy->open_flag );
    }
#else
    {
	DECLARE_WAITQUEUE(wait, current);
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue(&self->open_wait, &wait);
	if (LGET_BUDDY_OPEN_FLAG(major, minor) == 0) {
		/* Linuxξ硢RTOSΥץԤ */
		schedule();
	}
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&self->open_wait, &wait);
    }
#endif
    mutex_lock( &stat_mutex[major] );
    self->stat.open_count += 1;
    mutex_unlock( &stat_mutex[major] );

    self->pid = current->pid;
    self->comm = current->comm;
//    MOD_INC_USE_COUNT;
    filp->private_data = (void *)self;

    return ret;
}

/*
 * [Method] close
 * @inode   iΡ
 * @filp    ե빽¤
 * RETURN: 0
 */
static int devbrg_local_release(struct inode * inode, struct file * filp)
{
    struct iosc_port_t *self = filp->private_data;
    int major = self->major;

    iosc_message( self, "close", NULL );

    clean_packet_queue( &self->request );

    mutex_lock( &stat_mutex[major] );
    self->stat.release_count += 1;
    mutex_unlock( &stat_mutex[major] );

    mutex_lock( &open_mutex[major] );
    self->open_flag = 0;
    mutex_unlock( &open_mutex[major] );

    mutex_lock ( &getid_mutex[major] );
    self->lock_flag = 0;
    mutex_unlock ( &getid_mutex[major] );
    
//    MOD_DEC_USE_COUNT;
    return 0;
}

/*
 * [Method] read
 * @filp    ե빽¤
 * @buf     ɤ߹ǡǼХåե
 * @nbytes  ɤ߹ߥ
 * @ppos    ???ɤ߹ߥ֡
 * RETURN: 0ʾ塧ɤ߹
 *         -EFAULTɤ߽Ф顼
 */
static ssize_t devbrg_local_read(struct file *filp, char *buf, size_t nbytes, loff_t *ppos)
{
    struct iosc_port_t *self = filp->private_data;
    int major = self->major;
    ssize_t size = 0, ret;

    iosc_message( self, "read", "(buf=%08lX len=%d)", (unsigned long)buf, nbytes );

#if 0
    while(1){
        size = read_datapacket( &self->fifo, buf, nbytes );
        if( size < 0 )
        {
            wait_datapacket( &self->fifo );
        }
        else
        {
            break;
        }
    }
#else
    {
	DECLARE_WAITQUEUE(wait, current);
	set_current_state(TASK_INTERRUPTIBLE);
	add_wait_queue(&self->fifo.wait, &wait);
	while ((ret = read_datapacket(&self->fifo, buf + size, nbytes - size)) >= 0) {
		size += ret;
		if (size >= nbytes) {
			break;
		}
		schedule();
		set_current_state(TASK_INTERRUPTIBLE);
	}
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&self->fifo.wait, &wait);
    }
#endif

    mutex_lock( &stat_mutex[major] );
    self->stat.read_count++;
    if ( size >= 0)
    {
        self->stat.read_byte += size;
    }
    mutex_unlock( &stat_mutex[major] );
    return size;
}

/*
 * [Method] write
 * @filp    ե빽¤
 * @buf     񤭹ǡǼ줿Хåե
 * @nbytes  񤭹ߥ
 * @ppos    ???ʽ񤭹ߥ֡
 * RETURN: 0ʾ塧ʽ񤭹
 *         -ENOMEM˼
 *         -EFAULTɤ߽Ф顼
 */
static ssize_t devbrg_local_write(struct file *filp, const char *buf, size_t nbytes, loff_t *ppos)
{
    struct iosc_port_t *self = filp->private_data;
    int major = self->major;
    ssize_t size;
 	unsigned int listid = LISTID(self->major, self->minor, LID_QUE_FIFO, LID_PACKET_DAT , 0);

    iosc_message( self, "write", "(buf=%08lX len=%d)", (unsigned long)buf, nbytes );

#if 0
    size = write_datapacket( &self->buddy->fifo, (char *)buf, nbytes );
    wake_up_datapacket( &self->buddy->fifo );
#else
	size = write_datapacket(listid, (char *)buf, nbytes);
	LWAKE_UP_INTERRUPTIBLE(listid);
#endif

    mutex_lock( &stat_mutex[major] );
    self->stat.write_count += 1;
    if( size >= 0 )
    {
        self->stat.write_byte += size;
    }
    mutex_unlock( &stat_mutex[major] );
    return size;
}


/*
 * [Method] ioctl
 * @inode   iΡ
 * @filp    ե빽¤
 * @cmd     ޥ
 * @arg     ޥɰ
 * RETURN: 0ʾ塧
 *         -EIORTOSץ󤷤ƤʤΤ˽񤭹⤦Ȥ
 *         -ENOTTYޥɤ
 */
static int devbrg_local_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
    struct iosc_port_t *self = filp->private_data;
    int major = self->major;

    iosc_message( self, "ioctl", "(cmd=%08lX arg=%08lX)", (unsigned long)cmd, arg );
    mutex_lock( &stat_mutex[major] );
    self->stat.ioctl_count++;
    mutex_unlock( &stat_mutex[major] );

    switch( cmd )
    {   /* ߥ˥ɥ饤ж̥ޥ */
        case IOSC_ALLOC_MEM:
		return iosc_ioc_malloc(arg);
        case IOSC_FREE_MEM:
		return iosc_ioc_free(arg);
	case IOSC_ATTACH_MEM:
		return iosc_ioc_attach_mem(arg);
	case IOSC_DETACH_MEM:
		return iosc_ioc_detach_mem(arg);
        case IOSC_DEV_IOCPING:
            return iosc_ioc_ping_proc( self );
        case IOSC_DEV_IOCSIG:
            return iosc_ioc_sig_proc( self, (int)arg );
        case IOSC_DEV_IOCGET_ID:
            return iosc_ioc_getid_proc( self );
        case IOSC_DEV_IOCRELEASE_ID:
            return iosc_ioc_releaseid_proc( self );
        case IOSC_DEV_IOCGET_CMD:
            return iosc_ioc_getcmd_proc( self, (void *)arg );
        default:
            if( ( IOSC_GET_CMDDIR(cmd) == 0 ) &&
                ( IOSC_GET_CMDTYPE(cmd) == 0 ) &&
                ( IOSC_GET_CMDNR(cmd) == 11 ) )
            {
                return iosc_ioc_getdata_proc( self, (void *)arg, IOSC_GET_CMDSIZE(cmd) );
            }
            break;
    }

    switch( IOSC_GET_CMDDIR(cmd) )
    {   /* ޥ */
        case IOSC_IOC_RETURN:
            return iosc_cmd_return_proc( self, cmd, (void *)arg );
        case IOSC_IOC_ERROR:
            return iosc_cmd_error_proc( self, cmd, (void *)arg );
        case IOSC_IOC_DATARD:
            return iosc_cmd_data_read_proc( self, cmd, (void *)arg );
        case IOSC_IOC_DATAWR:
            return iosc_cmd_data_write_proc( self, cmd, (void *)arg );
        default:
            return iosc_cmd_send_proc( self, cmd, (void *)arg );
    }
}

static int iosc_ioc_malloc(int arg)
{
	struct iosc_mem_arg_t karg;
	int memid;
	if (copy_from_user(&karg, (void *)arg,
			   sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
	memid = LALLOC_MEM(karg.length);
	karg.memid = memid;
	if (copy_to_user((void*)arg, &karg,
			 sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
	if (memid < 0) {
		return -EPERM;
	}
	return 0;
}

static int iosc_ioc_free(int arg)
{
	struct iosc_mem_arg_t karg;
	if (copy_from_user(&karg, (void *)arg,
			   sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
	if (LFREE_MEM(karg.memid)) {
		return -EPERM;
	}
	return 0;
}

static int iosc_ioc_attach_mem(int arg)
{
	struct iosc_mem_arg_t karg;
	void *paddr;
	if (copy_from_user(&karg, (void *)arg,
			   sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
	paddr = LATTACH_MEM_PADDR(karg.memid);
	karg.paddr = paddr;
#ifdef IPRQ_L2_CACHEABLE
	karg.mmapflags = MAP_SHARED | MAP_UNCACHE_WB;
#else
	karg.mmapflags = MAP_SHARED | MAP_UNCACHE;
#endif
	if (copy_to_user((void*)arg, &karg,
			 sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
	if (!paddr) {
		return -EPERM;
	}
	return 0;
}

static int iosc_ioc_detach_mem(int arg)
{
	struct iosc_mem_arg_t karg;
	void *paddr = NULL;
	if (copy_from_user(&karg, (void *)arg,
			   sizeof(struct iosc_mem_arg_t))) {
		return -EFAULT;
	}
#if 1
	{
		struct vm_area_struct *vma;
		vma = find_vma(current->mm, (unsigned long)karg.memaddr);
		if (vma) {
			paddr = (void *)((vma->vm_pgoff << PAGE_SHIFT) +
					 ((unsigned long)karg.memaddr & ~PAGE_MASK));
		}
		/* Currently, paddr is not used in  LDETACH_MEM() ... */
	}
#endif
	if (LDETACH_MEM_PADDR(paddr)) {
		return -EPERM;
	}
	return 0;
}

/*
 * [SUB]
 *  ̿ΥǥХե뤬ץ󤵤Ƥ뤫ɤ֤
 */
static int iosc_ioc_ping_proc( struct iosc_port_t *self )
{
#if 0
    iosc_message( self, "ioctl", "IOSC_DEV_IOCPING" );
    if( self->buddy->open_flag ){ return 0; }
    else{ return -EAGAIN; }
#else
    if (LGET_BUDDY_OPEN_FLAG(self->major, self->minor)) {
	    return 0;
    } else {
	    return -EAGAIN;
    }
#endif
}

/*
 * [SUB]
 *  ץФơʥѥåȤ
 */
static int iosc_ioc_sig_proc( struct iosc_port_t *self, int arg )
{
    iosc_message( self, "ioctl", "IOSC_DEV_IOCSIG(arg=%x)", arg );
	return send_ctlpacket_self( &self->request, IOSC_DEV_IOCSIG, (void *)arg, sizeof(int) );
}

/*
 * [SUB]
 *  ץ󤵤Ƥʤեå
 */
static int iosc_ioc_getid_proc( struct iosc_port_t *self )
{
    int i, major = self->major;

    iosc_message( self, "ioctl", "IOSC_DEV_IOCGET_ID" );

	for (i = 0; i < iosc_dev[major]->n_minor; ++i) {
		mutex_lock(&getid_mutex[major]);
		if ((iosc_dev[major]->port[i].lock_flag == 0) &&
		    (iosc_dev[major]->port[i].open_flag == 0)) {
			iosc_dev[major]->port[i].lock_flag = current->pid;
			mutex_unlock(&getid_mutex[major]);
			return i;
		}
		mutex_unlock(&getid_mutex[major]);
	}
	return -EBUSY;
}

/*
 * [SUB]
 *  ץ󤵤ƤʤåƤեΥå
 */
static int iosc_ioc_releaseid_proc( struct iosc_port_t *self )
{
    int i, major = self->major;
    int cnt = 0;

    iosc_message( self, "ioctl", "IOSC_DEV_IOCRELEASE_ID" );

	for (i = 0; i < iosc_dev[major]->n_minor; ++i) {
		mutex_lock(&getid_mutex[major]);
		if ((iosc_dev[major]->port[i].lock_flag != 0) &&
		    (iosc_dev[major]->port[i].open_flag == 0)) {
			iosc_dev[major]->port[i].lock_flag = 0;
			cnt += 1;
		}
		mutex_unlock( &getid_mutex[major] );
    }
    if( cnt ){ return 0; }
    else{ return -EBUSY; }
}


/*
 * [SUB]
 *  ioctlȯԤ줿ޥɤ򥳥ޥɥѥåȤȤƼ
 */
static int iosc_ioc_getcmd_proc( struct iosc_port_t *self, struct iosc_read_packet_t *buf )
{
    struct iosc_read_packet_t tmp;
    iosc_message( self, "ioctl", "IOSC_DEV_IOCGET_CMD(buf=%p)", buf );
    receive_ctlpacket( &self->request, (void *)&tmp );
    if (copy_to_user(buf, &tmp, sizeof(struct iosc_read_packet_t))) {
	    return -EFAULT;
    }
    iosc_message( self, "ioctl", "get request-packet(cmd=%x len=%d arg=%x)\n", tmp.cmd, tmp.length, tmp.arg );
    return 0;
}

/*
 * [SUB]
 *  ioctlȯԤ줿ޥɤΥǡѥåȤ
 */
static int iosc_ioc_getdata_proc( struct iosc_port_t *self, void *arg, size_t size )
{
    iosc_message( self, "ioctl", "IOSC_DEV_IOCGET_DATA(arg=%p size=%d)", arg, size );
    read_datapacket( &self->request, (void *)arg, size );
    iosc_message( self, "ioctl", "get request-data-packet(data=%p len=%d)\n", arg, size );
    return 0;
}

/*
 * [SUB]
 *  ޥɤФֿԤ
 */
//
static int iosc_cmd_return_proc( struct iosc_port_t *self, unsigned int cmd, void *arg )
{
    size_t size = IOSC_GET_CMDSIZE(cmd);

    iosc_message( self, "ioctl", "IOSC_DEV_IOCRETURN(cmd=%x data=%p)", cmd, arg );
    iosc_message( self, "ioctl", "put reply-data-packet(data=%p len=%d)\n", arg, size );
#if 0
	write_datapacket( &self->buddy->reply, arg, IOSC_GET_CMDSIZE(cmd) );
#else
	write_datapacket(LISTID(self->major, self->minor, LID_QUE_REPLY, LID_PACKET_DAT, 0), arg, IOSC_GET_CMDSIZE(cmd));
#endif
    iosc_message( self, "ioctl", "put reply-packet(cmd=%x len=%d arg=%x)\n", cmd, size, arg );
#if 0
    return send_ctlpacket( &self->buddy->reply, cmd, arg, IOSC_GET_CMDSIZE(cmd) );
#else
	return send_ctlpacket(LISTID(self->major, self->minor, LID_QUE_REPLY, LID_PACKET_CTL, 0), cmd, arg, IOSC_GET_CMDSIZE(cmd));
#endif
}


/*
 * [SUB]
 *  ޥɤФ륨顼ֿԤ
 */
static int iosc_cmd_error_proc( struct iosc_port_t *self, unsigned int cmd, void *arg )
{
    iosc_message( self, "ioctl", "IOSC_DEV_IOCERROR(cmd=%x data=%p)", cmd, arg );
#if 0
    write_datapacket( &self->buddy->reply, arg, IOSC_GET_CMDSIZE(cmd) );
    send_ctlpacket( &self->buddy->reply, cmd, arg, IOSC_GET_CMDSIZE(cmd) );
#else
	write_datapacket(LISTID(self->major, self->minor, LID_QUE_REPLY, LID_PACKET_DAT, 0), arg, IOSC_GET_CMDSIZE(cmd));
#endif
	send_ctlpacket(LISTID(self->major, self->minor, LID_QUE_REPLY, LID_PACKET_CTL, 0), cmd, arg, IOSC_GET_CMDSIZE(cmd));
    return 0;
}

/*
 * [SUB]
 *  ǡޥɤ
 */
static int iosc_cmd_data_read_proc( struct iosc_port_t *self, unsigned int cmd, void *arg )
{
    iosc_message( self, "ioctl", "IOSC_IOC_DATARD(cmd=%x arg=%p)", cmd, arg );
    if (IOSC_GET_CMDNR(cmd) >= IOSC_MAX_BLOCK) {
	    return -EINVAL;
    }
#if 0
    return get_datablock( &self->buddy->block[IOSC_GET_CMDNR(cmd)], arg, IOSC_GET_CMDSIZE(cmd) );
#else
	return get_datablock(&self->block[IOSC_GET_CMDNR(cmd)], arg, IOSC_GET_CMDSIZE(cmd));
#endif
}

/*
 * [SUB]
 *  ǡǼޥɤ
 */
static int iosc_cmd_data_write_proc( struct iosc_port_t *self, unsigned int cmd, void *arg )
{
    iosc_message( self, "ioctl", "IOSC_IOC_DATAWR(cmd=%x arg=%p)", cmd, arg );
    if (IOSC_GET_CMDNR(cmd) >= IOSC_MAX_BLOCK) {
	    return -EINVAL;
    }
#if 0
    return set_datablock( &self->block[IOSC_GET_CMDNR(cmd)], arg, IOSC_GET_CMDSIZE(cmd) );
#else
	return set_datablock(LISTID(self->major, self->minor, LID_QUE_BLOCK, LID_PACKET_DAT, IOSC_GET_CMDNR(cmd)), arg, IOSC_GET_CMDSIZE(cmd));
#endif
}

/*
 * [SUB]
 *  ̾ޥɤ
 */
static int iosc_cmd_send_proc( struct iosc_port_t *self, unsigned int cmd, void *arg )
{
    size_t size = IOSC_GET_CMDSIZE(cmd);
    int dir = IOSC_GET_CMDDIR(cmd);
    int type = IOSC_GET_CMDTYPE(cmd);
    int nr = IOSC_GET_CMDNR(cmd);
    int result;
    int ret = 0;

    iosc_message( self, "ioctl", "USER COMMAND(cmd=%x type=%x nr=%d dir=%d size=%d arg=%p)", cmd, type, nr, dir, size, arg );

	if (type != self->type) {
		/* ꤬ǽʥޥɤǤϤʤ */
		return -ENOTTY;
	}

	mutex_lock (&self->mutex);
	if( ( size > 0 ) && ( ( dir == IOSC_IOC_WRITE ) || ( dir == IOSC_IOC_RW ) ) )
	{   /* size>0ǽ񤭹°Τ륳ޥ */
#if 0
		result = write_datapacket( &self->buddy->request, arg, size );
#else
		result = write_datapacket(LISTID(self->major, self->minor, LID_QUE_REQUEST, LID_PACKET_DAT, 0), arg, size);
#endif
		iosc_message( self, "ioctl", "put request-data-packet(data=%x len=%d)\n", (unsigned long)arg, size );
		if ( result < 0 )
		{ /* ǡ񤭹ߤ˼ */
			ret = result;
			goto unlock_mutex;
		}
	}

#if 0
    result = send_ctlpacket( &self->buddy->request, cmd, arg, size );
#else
	result = send_ctlpacket(LISTID(self->major, self->minor, LID_QUE_REQUEST, LID_PACKET_CTL, 0), cmd, arg, size);
#endif
    iosc_message( self, "ioctl", "put request-packet(cmd=%x len=%d arg=%x)\n", cmd, size, (unsigned long)arg );
    if ( result < 0 )
    {
        ret = result;
        goto unlock_mutex;
    }
    
    if( ( size > 0 ) && ( ( dir == IOSC_IOC_READ ) || ( dir == IOSC_IOC_RW ) ) )
    {
        struct iosc_read_packet_t packet;
        iosc_message( self, "ioctl", "wait reply-packet\n" );
        result = receive_ctlpacket( &self->reply, &packet );
        iosc_message( self, "ioctl", "get reply-packet(cmd=%x len=%d arg=%x)\n", packet.cmd, packet.length, packet.arg );
	if ( result < 0) {
		ret = result;
		goto unlock_mutex;
	}
		if( packet.length > 0 )
		{
			result = read_datapacket( &self->reply, arg, size );
			iosc_message( self, "ioctl", "get reply-data-packet(data=%x len=%d)\n", (unsigned long)arg, packet.length );
			if( result < 0 )
			{
				ret = result;
				goto unlock_mutex;
			}
		}
	}

unlock_mutex:
    mutex_unlock (&self->mutex);
    return ret;
}

/*
 * [Method] poll/select
 * @filp    ե빽¤
 * @wait    poll¤
 * RETURN: 0ʾ塧
 */
static int devbrg_local_poll(struct file *filp, struct poll_table_struct * wait)
{
    struct iosc_port_t *self = filp->private_data;
    int major = self->major;
    int status = 0;

    mutex_lock( &stat_mutex[major] );
    self->stat.poll_count++;
    mutex_unlock( &stat_mutex[major] );

    poll_wait(filp, &self->request.wait, wait);

    if (!list_empty(&self->request.datapacket_list) )
    {
        status |= POLLIN;
    }
    if (!list_empty(&self->request.ctlpacket_list))
    {
        status |= POLLPRI;
    }
#if 0
    if (list_empty(&self->buddy->request.ctlpacket_list))
    {
        status |= POLLOUT;
    }
#endif
    iosc_message( self, "poll", "(status=%x)", status );
    return status;
}

/*****************************************************************
 * other functions
 *****************************************************************/


static inline int noncached_address(unsigned long addr)
{
    return addr >= __pa(high_memory);
}

static void iosc_message( struct iosc_port_t *self, char *method, char *fmt, ... )
{
#ifndef NO_DEBUG_MSG
    va_list ap;
    char *str;
    	int major = self->major;

	printk(KERN_INFO "%s(%d)@%s#%s():", iosc_dev[major]->module_name, self->minor, "Linux", method);
    str = (char *)local_alloc_memory( sizeof(char)*1024 );
    if( str == NULL )
    {
        return ;
    }
    if( fmt != NULL )
    {
        va_start( ap, fmt );
        vsprintf( str, fmt, ap );
        va_end( ap );
        printk( "%s", str );
    }
    local_free_memory( str );
    printk( "\n" );
#endif
}

#if 0
IOSC_DEFINE_DEVICE_BRIDGE( frc, L_FRC_MAJOR, R_FRC_MAJOR, L_FRC_MAGIC, R_FRC_MAGIC, FRC_MINOR_NUM, FRC_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( gfx, L_GFX_MAJOR, R_GFX_MAJOR, L_GFX_MAGIC, R_GFX_MAGIC, GFX_MINOR_NUM, GFX_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( ipr, L_IPR_MAJOR, R_IPR_MAJOR, L_IPR_MAGIC, R_IPR_MAGIC, IPR_MINOR_NUM, IPR_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( mss, L_MSS_MAJOR, R_MSS_MAJOR, L_MSS_MAGIC, R_MSS_MAGIC, MSS_MINOR_NUM, MSS_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( od, L_OD_MAJOR, R_OD_MAJOR, L_OD_MAGIC, R_OD_MAGIC, OD_MINOR_NUM, OD_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( tcon, L_TCON_MAJOR, R_TCON_MAJOR, L_TCON_MAGIC, R_TCON_MAGIC, TCON_MINOR_NUM, TCON_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( tsd, L_TSD_MAJOR, R_TSD_MAJOR, L_TSD_MAGIC, R_TSD_MAGIC, TSD_MINOR_NUM, TSD_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vbi, L_VBI_MAJOR, R_VBI_MAJOR, L_VBI_MAGIC, R_VBI_MAGIC, VBI_MINOR_NUM, VBI_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vdx, L_VDX_MAJOR, R_VDX_MAJOR, L_VDX_MAGIC, R_VDX_MAGIC, VDX_MINOR_NUM, VDX_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vout, L_VOUT_MAJOR, R_VOUT_MAJOR, L_VOUT_MAGIC, R_VOUT_MAGIC, VOUT_MINOR_NUM, VOUT_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( exiv, L_EXIV_MAJOR, R_EXIV_MAJOR, L_EXIV_MAGIC, R_EXIV_MAGIC, EXIV_MINOR_NUM, EXIV_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( adeca, L_ADECA_MAJOR, R_ADECA_MAJOR, L_ADECA_MAGIC, R_ADECA_MAGIC, ADECA_MINOR_NUM, ADECA_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vop, L_VOP_MAJOR, R_VOP_MAJOR, L_VOP_MAGIC, R_VOP_MAGIC, VOP_MINOR_NUM, VOP_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( icc, L_ICC_MAJOR, R_ICC_MAJOR, L_ICC_MAGIC, R_ICC_MAGIC, ICC_MINOR_NUM, ICC_MMAP_BASE);

#else

IOSC_DEFINE_DEVICE_BRIDGE( frc, L_FRC_MAJOR, 0, L_FRC_MAGIC, 0, FRC_MINOR_NUM, FRC_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( ipr, L_IPR_MAJOR, 0, L_IPR_MAGIC, 0, IPR_MINOR_NUM, IPR_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( mss, L_MSS_MAJOR, 0, L_MSS_MAGIC, 0, MSS_MINOR_NUM, MSS_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( od, L_OD_MAJOR, 0, L_OD_MAGIC, 0, OD_MINOR_NUM, OD_MMAP_BASE);

#if defined(CONFIG_PROC_MN2WS0270)
IOSC_DEFINE_DEVICE_BRIDGE( tcon, L_TCON_MAJOR, 0, L_TCON_MAGIC, 0, TCON_MINOR_NUM, TCON_MMAP_BASE);
#endif /* defined(CONFIG_PROC_MN2WS0270) */

IOSC_DEFINE_DEVICE_BRIDGE( tsd, L_TSD_MAJOR, 0, L_TSD_MAGIC, 0, TSD_MINOR_NUM, TSD_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vout, L_VOUT_MAJOR, 0, L_VOUT_MAGIC, 0, VOUT_MINOR_NUM, VOUT_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( exiv, L_EXIV_MAJOR, 0, L_EXIV_MAGIC, 0, EXIV_MINOR_NUM, EXIV_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( exivext, L_EXIV_EXT_MAJOR, 0, L_EXIV_MAGIC, 0, EXIV_MINOR_NUM, EXIV_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( adeca, L_ADECA_MAJOR, 0, L_ADECA_MAGIC, 0, ADECA_MINOR_NUM, ADECA_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( vop, L_VOP_MAJOR, 0, L_VOP_MAGIC, 0, VOP_MINOR_NUM, VOP_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( icc, L_ICC_MAJOR, 0, L_ICC_MAGIC, 0, ICC_MINOR_NUM, ICC_MMAP_BASE);

IOSC_DEFINE_DEVICE_BRIDGE( sync, L_SYNC_MAJOR, 0, L_SYNC_MAGIC, 0, SYNC_MINOR_NUM, SYNC_MMAP_BASE);

#endif

/*****************************************************************
 * device initialize/exit functions
 *****************************************************************/

/*
 * static __init int iosc_dev_init(void)
 *-----------------------------------------------------------------------------
 * function: initialize modules
 * argument: nothing
 * return  :  0         : success
 *            under 0   : fail
 * comment :
 */
static __init int iosc_dev_init(void)
{
	init_frc();
	init_ipr();
	init_mss();
	init_od();
	init_tsd();
	init_vout();
	init_exiv();
	init_exivext();
	init_adeca();
	init_vop();
	init_icc();
	init_sync();
#if defined(CONFIG_PROC_MN2WS0270)
	init_tcon();
#endif /* defined(CONFIG_PROC_MN2WS0270) */
	
	init_iprq();
	iosc_share_init();
	intdev_local_init();

	return 0;
}

/*
 * static __exit void iosc_dev_exit(void)
 *-----------------------------------------------------------------------------
 * function: exit modules
 * argument: nothing
 * return  : nothing
 * comment :
 */
static __exit void iosc_dev_exit(void)
{
	exit_frc();
	exit_ipr();
	exit_mss();
	exit_od();
	exit_tsd();
	exit_vout();
	exit_exiv();
	exit_exivext();
	exit_adeca();
	exit_vop();
	exit_icc();
	exit_sync();
#if defined(CONFIG_PROC_MN2WS0270)
	exit_tcon();
#endif /* defined(CONFIG_PROC_MN2WS0270) */

	iosc_share_exit();
	intdev_local_exit();
	exit_iprq();
}

module_init(iosc_dev_init);
module_exit(iosc_dev_exit);

/* End of dev-internal.c */

#endif /* DEV_INTERNAL_C */

#ifdef MODULE_LICENSE
MODULE_LICENSE("Proprietary");
#endif
